/**************************************************************************************************
 * SMAC implementation.
 * 
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2010 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
***************************************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
***********************************************************************************************//*!
**************************************************************************************************/

#include "SMAC.h"
#include "Phy.h"
#include "EmbeddedTypes.h"
#include "TransceiverDrv.h"
#include "PhyPibExtended.h"
#include "Interrupt.h"
#include "SMAC_Config.h"

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
 bool_t smacStandalone = TRUE;           

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
static smacStates_t    smacState;
volatile static prssPacketPtr_t smacProccesPacketPtr;
static phyRxParams_t   smacLastDataRxParams;
static uint32_t        smacTimeout;
//static uint8_t         smacEdValue;           //Not used
//static uint8_t      smacClearestChann;        //Not used
static smacPacket_t smacPacketConfig;
flagsRegister_t smacFlags;

const uint8_t au8SmacVersion[7] = 
{
  mMajorSmacV_c, 
  mMiddleSmacV_c, 
  mMinorSmacV_c, 
  mMonthSmacV_c,
  mDaySmacV_c, 
  mYearSmacV_c, 
  mConsSmacV_c
};
const uint8_t au8PhyVersion[7] = 
{
  mMajorPhyV_c, 
  mMiddlePhyV_c, 
  mMinorPhyV_c, 
  mMonthPhyV_c,
  mDayPhyV_c, 
  mYearPhyV_c, 
  mConsPhyV_c
};

smacRFConstants_t const smacRFConstants_FreqBand_863__870MHz[] =
{
// 863 MHz, Operating Mode 1
  {
  //channel center frequency 0
  868300000,         
  //channel spacing
  200000,
  //total number of channels
  34,
  //Radio bit rate register
  Bitrate_4800,
  //Radio frequency deviation register
  Fdev_5000,    
  //number of RSSI measurements
  4,//TODO
  //Radio RxBw register
  (DccFreq_5 | RxBw_10400),
  //Radio RxAfcBw register
  (DccFreq_5 | RxBw_10400),
  //Radio modulation parameters
  (DataModul_Modulation_Fsk | DataModul_ModulationShaping_NoShaping),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
  (160 >> 1)
  },
// 863 MHz, Operating Mode 2
  {
  //channel center frequency 0
    863225000,
  //channel spacing
    400000,
  //total number of channels
    17,
  //Radio bit rate register
    Bitrate_100000,
  //Radio frequency deviation register
    Fdev_50000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_166700),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_166700),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (167 >> 1)
  }
};

smacRFConstants_t const smacRFConstants_FreqBand_902__928MHz[] =
{
	//915 MHz, Operating Mode 1
	{
	//channel center frequency 0
	902500000,
	//channel spacing
	1000000,
	//total number of channels
	25,
	//Radio bit rate register
	Bitrate_50000,
	//Radio frequency deviation register
	Fdev_170000,
	//number of RSSI measurements
	4,//TODO
	//Radio RxBw register
	(DccFreq_2 | RxBw_250000),
	//Radio RxAfcBw register
	(DccFreq_2 | RxBw_250000),
	//Radio modulation parameters
	(DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_1),
	//CCA threshold
	//Must be 10 dB above receiver sensitivity multiplied by 2
	//Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
	(160 >> 1)
     },
//915 MHz, Operating Mode 2
  {
  //channel center frequency 0
    902200000,
  //channel spacing
    2000000,
  //total number of channels
    12,
  //Radio bit rate register
    Bitrate_150000,
  //Radio frequency deviation register
    Fdev_170000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_2 | RxBw_250000),
  //Radio RxAfcBw register
    (DccFreq_2 | RxBw_250000),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (160 >> 1)
  },
//915 MHz, Operating Mode 3
  {
  //channel center frequency 0
    902400000,
  //channel spacing
    2000000,
  //total number of channels
    12,
  //Radio bit rate register
    Bitrate_200000,
  //Radio frequency deviation register
    Fdev_180000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_2 | RxBw_333300),
  //Radio RxAfcBw register
    (DccFreq_2 | RxBw_333300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (173 >> 1)
  }
};

/*This band has the settings for the Americas cfg*/
smacRFConstants_t const smacRFConstants_FreqBand_950__958MHz[] =
{
//950 MHz, Operating Mode 1
  {
  //channel center frequency 0
	951000000,
	//channel spacing
	200000,
	//total number of channels
	34,
	//Radio bit rate register
	Bitrate_50000,
	//Radio frequency deviation register
	Fdev_100000,
	//number of RSSI measurements
	4,//TODO
	//Radio RxBw register
	(DccFreq_7 | RxBw_500000),
	//Radio RxAfcBw register
	(DccFreq_7 | RxBw_500000),
	//Radio modulation parameters
	(DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_1),
	//CCA threshold
	//Must be 10 dB above receiver sensitivity multiplied by 2
	//Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
	(160 >> 1)
  },
  //Dummy
  {
	  //channel center frequency 
	      0,
	    //channel spacing
	      0,
	    //total number of channels
	      0,
	    //Radio bit rate register
	      0,
	    //Radio frequency deviation register
	      0,
	    //number of RSSI measurements
	      0,//TODO
	    //Radio RxBw register
	      0,
	    //Radio RxAfcBw register
	      0,
	    //Radio modulation parameters
	      0,
	    //CCA threshold
	      0
  },
//950 MHz, Operating Mode 3
  {
  //channel center frequency 0
    951200000,
  //channel spacing
    600000,
  //total number of channels
    11,
  //Radio bit rate register
    Bitrate_200000,
  //Radio frequency deviation register
    Fdev_100000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_333300),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_333300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (173 >> 1)
  }

/*NOTE: This band has the settings for the 902-928 Japan Band*/
/*
smacRFConstants_t const smacRFConstants_FreqBand_950__958MHz[] =
{
	//920 MHz, Operating Mode 1 50kbps
	{
        //channel center frequency 0
        920600000,
        //channel spacing
        200000,
        //total number of channels
        62,  // Will use Max ch61
        //Radio bit rate register
        Bitrate_50000,
        //Radio frequency deviation register
        Fdev_25000,
        //number of RSSI measurements
        4,//TODO
        //Radio RxBw register
        0xB2,  //Ori (DccFreq_2 | RxBw_83300),
        //Radio RxAfcBw register
        0xB2, //Ori (DccFreq_4 | RxBwAFC_83300),
        //Radio modulation parameters
	(DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_1),
	//CCA threshold
	//Must be 10 dB above receiver sensitivity multiplied by 2
	//Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
	(160 >> 1)
  },
  //920 MHz, Operating Mode 2 100kbps
  {
      //channel center frequency 0
          92060000,
       //channel spacing
         400000,
       //total number of channels
          61,  // Will use Max ch60
       //Radio bit rate register
         Bitrate_100000,
       //Radio frequency deviation register
         Fdev_50000,
       //number of RSSI measurements
         4,//TODO
       //Radio RxBw register
         0xB1, //Ori 0x42,
       //Radio RxAfcBw register
         0xB1, // =0x8b
       //Radio modulation parameters
          (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_1),
        //CCA threshold
        //Must be 10 dB above receiver sensitivity multiplied by 2
        //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
          (160 >> 1)
  },
//920 MHz, Operating Mode 3 200kbps
  {
      //channel center frequency 0
          92060000,
       //channel spacing
         600000,
       //total number of channels
          61,  // Will use Max ch60
       //Radio bit rate register
         Bitrate_200000,
       //Radio frequency deviation register
         Fdev_50000,
       //number of RSSI measurements
         4,//TODO
       //Radio RxBw register
         0xB1, //Ori 0x42,
       //Radio RxAfcBw register
         0xB1, // =0x8b
       //Radio modulation parameters
          (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_1),
        //CCA threshold
        //Must be 10 dB above receiver sensitivity multiplied by 2
        //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
          (160 >> 1)
  }
*/
};


/************************************************************************************
*************************************************************************************
* Interface functions
*************************************************************************************
************************************************************************************/


/***********************************************************************************/
/******************************** SMAC Data primitives *****************************/
/***********************************************************************************/

/************************************************************************************
* MLMERadioInit
* 
* This function initializes the Radio parameters.
*
************************************************************************************/
smacErrors_t MLMERadioInit(void)
{
  
if(smacStandalone)
{
	gPhyPib.mPIBphyFSKScramblePSDU = TRUE;
	gPhyPib.mPIBphyTransmitPower = 0x1F;
	gPhyPib.mPIBphyFSKPreambleRepetitions = 3;
	gPhyPib.mPIBphyCurrentChannel = 0;
}
  
  
  /*Initialize PHY */
if(smacStandalone)
{
	MKW01Drv_RFdefaultInit();
	PhyPib_RFUpdatePowerStep();
}  
  MKW01Drv_WriteRegister(MKW01_Reg_PacketConfig1, 0xD8);
  MKW01Drv_WriteRegister(MKW01_Reg_PacketConfig2, 0x00);
  MKW01Drv_WriteRegister(MKW01_Reg_TestDagc, 0x10);  
  PhyPib_RFUpdateDCfreeEncoding();
  PhyPib_RFUpdatePreambleLength();     
  /* Radio payload length initialization */
  MKW01Drv_WriteRegister(MKW01_Reg_PayloadLength, (uint8_t) 255);  //max length in rx
  if(smacStandalone)
  {
	  PhyPib_RFUpdateDCfreeEncoding();
  	  PhyPib_RFUpdatePreambleLength();
  	  PhyPib_RFUpdatePowerStep();
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue1, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue2, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue3, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue4, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue5, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue6, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue7, 0x01 );
  	  MKW01Drv_WriteRegister(MKW01_Reg_SyncValue8, 0x01 );  
  }
  PhyTimerStart();                                                              //Timer 2
  PhyEventTimeoutInit();
  PhyEventTriggerInit();

  MKW01xDrv_IRQ_PortConfig();                                                   //At this point only KBI2SC, KBIE=1 is enabled (No edge select nor Pin Interrupt enabled) (DIO0, DIO1)
  MKW01Drv_CCA_Enable();                                                        //At this point IRQSC, IRQIE=1, Rising edge/high level (DIO4)
                                               
  EnableInterrupts();
  smacState= mSmacStateIdle_c;
  smacLastDataRxParams.linkQuality = 0;
  smacLastDataRxParams.timeStamp = 0;
  //smacEdValue = 0;							//Not used
  //smacClearestChann = 0xFF;                                           //Not used /* No channel */ 
  smacFlags.entryRegister = 0;         
  /* it means:
              mSmacInitialized        = FALSE;
              mSmacTimeoutAsked       = FALSE;
              mSmacPerformingCCAScan  = FALSE;
              mSmacPerformingEDScan   = FALSE;
              mSmacPerformingED       = FALSE;
  */
        
#if(TRUE == smacInitializationValidation_d)
    mSmacInitialized = TRUE;  
#endif
    return gErrorNoError_c;

}


/************************************************************************************
* MCPSDataRequest
* 
* This data primitive is used to send an over the air packet. This is an asynchronous 
* function,  it means it asks SMAC to transmit one OTA packet,  but when the function 
* returns it is not sent already.
*
************************************************************************************/
smacErrors_t MCPSDataRequest
(
txPacket_t *psTxPacket        //IN:Pointer to the packet to be transmitted
)
{  
  uint8_t u8PhyRes = 0; 
  
  
  
#if(TRUE == smacParametersValidation_d)
  uint8_t u8MaxLen=0;
  
  u8MaxLen = gMaxSmacSDULength_c;

  if((NULL == psTxPacket) || (u8MaxLen < psTxPacket->u8DataLength) || (gMinSmacSDULength_c > psTxPacket->u8DataLength))
  {
    return gErrorOutOfRange_c;
  }  
#endif         /* TRUE == smacParametersValidation_d */

#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif      /* TRUE == smacInitializationValidation_d */
 
  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
  
  psTxPacket->u8DataLength = psTxPacket->u8DataLength + gSmacHeaderBytes_c; 
  
  smacProccesPacketPtr.smacTxPacketPointer  = SmacFormatTxPacket(psTxPacket);

  u8PhyRes = PhyDataRequest(smacProccesPacketPtr.smacTxPacketPointer, gImmediateAction_c);

  psTxPacket->u8DataLength = psTxPacket->u8DataLength - gSmacHeaderBytes_c;
  
  if(!u8PhyRes)
  {
    smacState= mSmacStateTransmitting_c; 
    return gErrorNoError_c;
  }
  else
  {
    return gErrorNoResourcesAvailable_c;
  }
  
 

}

/************************************************************************************
* MLMERXEnableRequest
* 
* Function used to place the radio into receive mode 
*
************************************************************************************/
smacErrors_t MLMERXEnableRequest
(
rxPacket_t *gsRxPacket, //OUT: Pointer to the structure where the reception results 
                        //     will be stored.
uint32_t u32Timeout     //IN:  32-bit timeout value, this is directly the value that 
                        //     is stored on the Radio's timer register.
)
{

  uint8_t u8PhyRes = 0; 
  phyPacket_t * pAuxPacket;

#if(TRUE == smacParametersValidation_d)
  uint8_t u8MaxLen=0;
	  
  u8MaxLen = gMaxSmacSDULength_c;
#endif         /* TRUE == smacParametersValidation_d */
  

    pAuxPacket = (phyPacket_t *)&gsRxPacket->u8DataLength; 

#if(TRUE == smacParametersValidation_d)
  if((NULL == gsRxPacket) || (u8MaxLen < gsRxPacket->u8MaxDataLength))
  {
    return gErrorOutOfRange_c;
  }
#endif     /* TRUE == smacParametersValidation_d */

#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
                
  u8PhyRes = PhyPlmeRxRequest(pAuxPacket, &smacLastDataRxParams, gImmediateAction_c);

  if(!u8PhyRes)
  {
  
    if(u32Timeout)
    {
      smacTimeout = u32Timeout; 
      SmacSetRxTimeout(smacTimeout);
      mSmacTimeoutAsked = TRUE;
    }
    else
    {
      mSmacTimeoutAsked = FALSE;
    }

    gsRxPacket->rxStatus = rxProcessingReceptionStatus_c;
    smacProccesPacketPtr.smacRxPacketPointer  = gsRxPacket;

    smacState= mSmacStateReceiving_c; 
    return gErrorNoError_c;
  }
  else
  {
    return gErrorNoResourcesAvailable_c;
  }
}

/************************************************************************************
* MLMESetPreambleLength
* 
* Function used to change the preamble size in the OTA Packet 
*
************************************************************************************/
smacErrors_t MLMESetPreambleLength
(
uint16_t u16preambleLength
)
{

	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
                
  gPhyPib.mPIBphyFSKPreambleRepetitions = u16preambleLength;
  
  smacPacketConfig.u16PreambleLength = u16preambleLength;
  
  PhyPib_RFUpdatePreambleLength();
    
  return gErrorNoError_c;
 
}

/************************************************************************************
* MLMESetSyncWordSize
* 
* Function used to change the synchronization word size. Values from 0 to 8 required. 
* 
* Inputs      : 
* 	SyncConfig_SyncSize_1             
*	SyncConfig_SyncSize_2              
*	SyncConfig_SyncSize_3               
*	SyncConfig_SyncSize_4               
* 	SyncConfig_SyncSize_5                
* 	SyncConfig_SyncSize_6                
*	SyncConfig_SyncSize_7              
*  	SyncConfig_SyncSize_8    
*
************************************************************************************/
smacErrors_t MLMESetSyncWordSize
(
uint8_t u8syncWordSize
)
{
	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
  
  (void)PhyPib_SetSyncWordSize(u8syncWordSize);
  
  smacPacketConfig.u8SyncWordSize = (u8syncWordSize >> 3) + 1;
  
  return gErrorNoError_c;
 
}

/************************************************************************************
* MLMESetSyncWordValue
* 
* Function used to change the synchronization word value. 
*     
*
************************************************************************************/
smacErrors_t MLMESetSyncWordValue
(
uint8_t *u8syncWordValue
)
{
	uint8_t syncWordSizeTemp = smacPacketConfig.u8SyncWordSize;
	uint8_t syncValueReg = MKW01_Reg_SyncValue1;
	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
  
  smacPacketConfig.u8SyncWordValue = u8syncWordValue;
  
  while (syncWordSizeTemp--)
  {
	  PhyPib_SetSyncWordValue(syncValueReg, (uint8_t)*u8syncWordValue);
	  syncValueReg++;
	  u8syncWordValue++;
  }
  while(MKW01_Reg_SyncValue8 < syncValueReg)
  {
	  PhyPib_SetSyncWordValue(syncValueReg, 0x00);
	  syncValueReg++;
  }
  
  return gErrorNoError_c;
 
}

/************************************************************************************
* MLMEPacketConfig
* 
*  
*
************************************************************************************/
smacErrors_t MLMEPacketConfig
(
packetConfig_t *pPacketCfg
)
{
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }	
  MLMESetSyncWordValue(pPacketCfg->pu8SyncWord);
  MLMESetSyncWordSize(pPacketCfg->u8SyncWordSize);
  MLMESetPreambleLength(pPacketCfg->u16PreambleSize);
  return gErrorNoError_c;	
}

/************************************************************************************
* MLMESetChannelRequest
* 
*  
*
************************************************************************************/
smacErrors_t MLMESetChannelRequest
(
 channels_t newChannel
)
{
uint8_t errorVal;	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */
  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }	
  errorVal = PhyPib_SetCurrentChannel(newChannel);
  switch (errorVal)
  {
  case  gPhyPibBusyTransceiver_c:
   return gErrorBusy_c;
   break;
   
  case gPhyPibInvalidParameter_c:	 
   return gErrorOutOfRange_c;
  break;
  
  case gPhyPibSuccess_c: 
   return gErrorNoError_c;
  break;
  
  default:
    return gErrorOutOfRange_c;  //AC added, to get rid of warning Pe940
    break;
  }
}

/************************************************************************************
* MLMESetFreqBand
* 
*  
*
************************************************************************************/
smacErrors_t MLMESetFreqBand
(
	smacFrequencyBands_t freqBand,
	smacRFModes_t phyMode
)
{	
   
	phyMode_t *pPhyModeSupported = (phyMode_t *) &gPhyPib.mPIBphyModeSupported;
      	phyMode_t *pPhyCurrentMode = (phyMode_t *) &gPhyPib.mPIBphyCurrentMode;
                   
	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */
  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }	
  
  switch(freqBand)
  {
    case g863__870MHz_c:
            pPhyCurrentMode->bitAccess.freqBand = g863__870MHz_c;        
            pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode1_c);
            pPhyCurrentMode->bitAccess.phyMode = (phyMode);
  
            gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &smacRFConstants_FreqBand_863__870MHz[phyMode-1];
            (void)PhyPib_SetCarrierFreq(0xD91332);                              //868.3MHz for a 32MHz XTAL source
            //(void)PhyPib_SetCarrierFreq(0xE78BF2);                            //868.3MHz for a 30MHz XTAL source
            break;
	
      case g950__958MHz_c:                                      
    	  pPhyCurrentMode->bitAccess.freqBand = g950__958MHz_c;        
    	  pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode2_c);
    	  pPhyCurrentMode->bitAccess.phyMode = (phyMode);
        
    	  gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &smacRFConstants_FreqBand_950__958MHz[phyMode-1];    	  
    	 (void)PhyPib_SetCarrierFreq(0xED8027);                                 //920 MHz for a 32MHz XTAL source
         // (void)PhyPib_SetCarrierFreq(0xF55555);                              //920MHz for a 30MHz XTAL source
    	  break;
    	  
      case g902__928MHz_c:
     	  pPhyCurrentMode->bitAccess.freqBand = g902__928MHz_c;               
          pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode1_c | gPhyMode2_c );
          pPhyCurrentMode->bitAccess.phyMode = (phyMode);                    
          
    	  gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &smacRFConstants_FreqBand_902__928MHz[phyMode-1];
    	  (void)PhyPib_SetCarrierFreq(0xE4C026);                                //915 MHz for a 32MHz XTAL source
          //(void)PhyPib_SetCarrierFreq(0xF40000);                              //915 MHz for a 30MHz XTAL source
    	  break; 
          
      default:
     	  pPhyCurrentMode->bitAccess.freqBand = g902__928MHz_c;               
          pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode1_c | gPhyMode2_c );
          pPhyCurrentMode->bitAccess.phyMode = (phyMode);                    
          
    	  gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &smacRFConstants_FreqBand_902__928MHz[phyMode-1];
    	  (void)PhyPib_SetCarrierFreq(0xE4C026);                                //915 MHz for a 32MHz XTAL source
          //(void)PhyPib_SetCarrierFreq(0xF40000);                              //915 MHz for a 30MHz XTAL source
    	  break; 
    }

    PhyPib_RFUpdateModulationParameters();
    PhyPib_RFUpdateDCfreeEncoding();
    PhyPib_RFUpdatePreambleLength();                  
  
    return gErrorNoError_c;
}

/************************************************************************************
* MLMESetChannelRequest
* 
*  
*
************************************************************************************/
channels_t MLMEGetChannelRequest
(
void 
)
{
 channels_t currentChannel;
	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }	
  PhyPib_Get(gPhyPibCurrentChannel_c,  &currentChannel);  
  return currentChannel;	
}

/************************************************************************************
* MLMERssi
* 
*  
*
************************************************************************************/
uint8_t MLMERssi
(
void 
)
{
 uint8_t rssiVal;
	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
  
  rssiVal = PhyPib_GetRssi();
  return rssiVal;
}


/************************************************************************************
* MLMESetInterPacketRxDelay
* 
*  
*
************************************************************************************/
smacErrors_t MLMESetInterPacketRxDelay
(
uint8_t u8InterPacketRxDelay
)
{

	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif     /* TRUE == smacInitializationValidation_d */

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
  
  if (gPhyPibSuccess_c != PhyPib_SetInterPacketRxDelay(u8InterPacketRxDelay))
  {
	  return gErrorOutOfRange_c;
  }
  
    
  return gErrorNoError_c;
 
}


/************************************************************************************
* MLMERXDisableRequest
* 
* Returns the radio to idle mode from receive mode.
*
************************************************************************************/
smacErrors_t MLMERXDisableRequest(void)
{
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return gErrorNoValidCondition_c;
  }
#endif
  if((mSmacStateReceiving_c != smacState) && (mSmacStateIdle_c != smacState))
  {
    return gErrorNoValidCondition_c;
  }
  
  PhyPlmeForceTrxOffRequest();
  mSmacTimeoutAsked = FALSE;
  
  smacState= mSmacStateIdle_c;
  return gErrorNoError_c;

}
 
/************************************************************************************
* MLMELinkQuality
* 
* This  function  returns  an  integer  value  that is the link quality from the last 
* received packet of the form:  dBm = (-Link Quality/2).
*
************************************************************************************/
uint8_t MLMELinkQuality(void)
{
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
  {
    return 0;
  }
#endif
  return smacLastDataRxParams.linkQuality;
}

/************************************************************************************
* MLMEPAOutputAdjust
* 
*
************************************************************************************/
smacErrors_t MLMEPAOutputAdjust
( 
	uint8_t u8PaValue
)
{
uint8_t errorVal;		
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 errorVal = PhyPib_SetCurrentTransmitPower(u8PaValue);
	 switch (errorVal)
	 {
	 case  gPhyPibBusyTransceiver_c:
	  return gErrorBusy_c;
	  break;
	  
	 case gPhyPibInvalidParameter_c:	 
	  return gErrorOutOfRange_c;
	 break;
	 
	 case gPhyPibSuccess_c: 
	  return gErrorNoError_c;
	 break;
         	
        default:
	return gErrorOutOfRange_c;  	//AC added, to get rid of warning Pe940
	break;
 }
}

/************************************************************************************
* MLMEStandbyRequest
* 
* This  function  returns  an  integer  value  that is the link quality from the last 
* received packet of the form:  dBm = (-Link Quality/2).
*
************************************************************************************/
smacErrors_t MLMEStandbyRequest( void )
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
  MKW01Drv_RadioStandByReq(0);
  return gErrorNoError_c;
}
	 
/************************************************************************************
* MLMESleepRequest
* 
* This  function  returns  an  integer  value  that is the link quality from the last 
* received packet of the form:  dBm = (-Link Quality/2).
*
************************************************************************************/
smacErrors_t MLMESleepRequest( void )
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
  MKW01Drv_RadioSleepReq();
  return gErrorNoError_c;
}

/************************************************************************************
* MLMEListenRequest
* 
* This  function  returns  an  integer  value  that is the link quality from the last 
* received packet of the form:  dBm = (-Link Quality/2).
*
************************************************************************************/
smacErrors_t MLMEListenRequest( listenResolution_t listenResolution, uint8_t listenCoef)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 PhyPib_SetListenResRx((uint8_t)listenResolution);
 PhyPib_SetListenCoefRx(listenCoef);
 
 PhyPib_SetOperationMode(OpMode_StandBy);
 PhyPib_SetOperationMode(OpMode_Listen_On); 
  return gErrorNoError_c;
}

/************************************************************************************
* MLMEScanRequest
* 
* This  function  returns  the RSSI value of the channel passes as a parameter  
* 
*
************************************************************************************/
smacErrors_t MLMEScanRequest
(
 channels_t u8ChannelToScan,
 uint8_t *u8ChannelScanResult		
)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 MLMESetChannelRequest(u8ChannelToScan);
 *u8ChannelScanResult = MLMERssi(); 
 return gErrorNoError_c;
}

/************************************************************************************
* SMACDisableInterrupts 
* 
* This  function disable all the MCU interrupts used by the radio (IRQ, KBI)
* 
*
************************************************************************************/
void SMACDisableInterrupts
(
void	
)
{	
MKW01Drv_IRQ_Disable();
MKW01Drv_CCA_Disable();
}

/************************************************************************************
* SMACEnableInterrupts 
* 
* This  function enable all the MCU interrupts used by the radio (IRQ, KBI)
* 
* 
*
************************************************************************************/
void SMACEnableInterrupts
(
void	
)
{	

  MKW01xDrv_IRQ_PortConfig();                                                   //At this point only DIO0, DIO1, DIO4 are enabled (No edge select nor Pin Interrupt enabled)Can Not cause interrupt yet

}

/************************************************************************************
* MLMEGainAdjust 
* 
* This function adjusts the LNA gain 
* 
*
************************************************************************************/
smacErrors_t MLMELnaGainAdjust
(
 lnaGainValues_t  gainValue	
)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 
 PhyPib_SetLnaGain((uint8_t)gainValue);
 return gErrorNoError_c;
}

/************************************************************************************
* MLMEEnablePABoost
* 
* This function enables the PA1 and PA2 or both 
* 
*
************************************************************************************/
smacErrors_t MLMEEnablePABoost 
(
 uint8_t u8PABoostCfg
)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 if ( gDisablePA_Boost_c == u8PABoostCfg ||
	  gEnablePA2_Boost_c == u8PABoostCfg || 
	  gEnablePA1_Boost_c == u8PABoostCfg || 
	  gEnablePABoth_Boost_c == u8PABoostCfg )
 {
  PhyPib_PABoostCfg (u8PABoostCfg);
  return gErrorNoError_c;
 }
 else
  return gErrorNoValidCondition_c;
}

/************************************************************************************
* MLMEDisablePABoost
* 
*This function disables the PA modes and sets it to PA0  (normal).
* 
*
************************************************************************************/
smacErrors_t MLMEDisablePABoost 
(
 void	
)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 PhyPib_PABoostCfg (gDisablePA_Boost_c);
 return gErrorNoError_c;
}

/************************************************************************************
* MLMEPhySoftReset
* 
* This function performs a software reset on the radio, PHY and SMAC state machines.
* 
*
************************************************************************************/
smacErrors_t MLMEPhySoftReset
(
 void	
)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 PhyPlmeForceTrxOffRequest();
 smacState= mSmacStateIdle_c; 
 return gErrorNoError_c;
}

/************************************************************************************
* MLMESetClockRate
* 
* This function is called to set the desired clock out from radio.
* 
*
************************************************************************************/
smacErrors_t MLMESetClockRate
(
 clkoFrequency_t clkFreq
)
{	
#if(TRUE == smacInitializationValidation_d)
 if(FALSE == mSmacInitialized)
    {
    return gErrorNoValidCondition_c;
    }
#endif /* TRUE == smacInitializationValidation_d */
 
 if(mSmacStateIdle_c != smacState)
 {
   return gErrorBusy_c;
 }
 MKW01Drv_ConfigureCLKO(clkFreq);
 return gErrorNoError_c;
}

/************************************************************************************
* MLMEGetRficVersion
* 
* This function is used to read the version number of the software modules inside the SMAC platform.
* 
*
************************************************************************************/
smacErrors_t MLMEGetRficVersion
(
 versionedEntity_t entity, 
 uint8_t *Buffer
)
{	
 uint8_t u8i; 
 
#if(TRUE == smacParametersValidation_d)
  if(gMaxVersionedEntity_c <= entity)
  {
    return gErrorOutOfRange_c;
  }
#endif

  if(mSmacStateIdle_c != smacState)
  {
    return gErrorBusy_c;
  }
  
  if(gSwSmacVersion_c == entity)
  {
    for(u8i=0; u8i < gSwSmacVersionLength_c; u8i++)
    {
      *Buffer++ = au8SmacVersion[u8i];
    }
  }
  else if (gSwPhyVersion_c == entity)
  {
	for(u8i=0; u8i < gSwPhyVersionLength_c; u8i++)
	{
	 *Buffer++ = au8PhyVersion[u8i];
	}    
  }

  return gErrorNoError_c;
}	

/************************************************************************************
* XCVRContReset
* 
* This function asserts the reset line of the transceiver.
* 
*
************************************************************************************/
void XCVRContReset
(
 void
)
{	
  /* asset transceiver reset line */
  MKW01Drv_AssertReset();
}

/************************************************************************************
* XCVRRestart
* 
* This function de-asserts the reset line of the transceiver.
* 
*
************************************************************************************/
void XCVRRestart
(
void		
)
{
  /* Deassert transceiver reset line */
  MKW01Drv_DeassertReset(); 
}

/*********************************************************/
void SmacSetRxTimeout(uint32_t timeoutSymbols)
{
	PhyTimeSetEventTimeout(timeoutSymbols);
}

/*********************************************************/
phyPacket_t * SmacFormatTxPacket(txPacket_t *psTxPacket)
{
  phyPacket_t * packetToPhy;
  
  packetToPhy = (phyPacket_t *)psTxPacket;
  
  return packetToPhy;
}

/***********************************************************************************/
/****************************** PHY Callback Functions *****************************/
/***********************************************************************************/

void    PhyPdDataConfirm(void)
{
	if(mSmacStateTransmitting_c == smacState)
	{   
	    MCPSDataConfirm(txSuccessStatus_c);
	    smacState= mSmacStateIdle_c;
	}
}

void    PhyPdDataIndication(void)
{
	 uint8_t  u8PhyRes;
	 uint8_t u8destAddress;
	 
	 u8destAddress = smacProccesPacketPtr.smacRxPacketPointer->smacPdu.u8DestAddress;
	 
	 smacProccesPacketPtr.smacRxPacketPointer->u8DataLength-=gSmacHeaderBytes_c;
	 
	 if((smacProccesPacketPtr.smacRxPacketPointer->u8DataLength > smacProccesPacketPtr.smacRxPacketPointer->u8MaxDataLength)
			 || (!(IsBroadcastMessage(u8destAddress)) && !( gNodeAddress_c == u8destAddress )))
	 {                                                                                      
		 u8PhyRes = PhyPlmeRxRequest((phyPacket_t *)&(smacProccesPacketPtr.smacRxPacketPointer->u8DataLength)-gSmacHeaderBytes_c, &smacLastDataRxParams, gImmediateAction_c);
		 if(!u8PhyRes){
			 if(mSmacTimeoutAsked)
			 {
				 SmacSetRxTimeout((zbClock24_t)smacTimeout);
			 }
		 }else{
			 smacProccesPacketPtr.smacRxPacketPointer->rxStatus = rxAbortedStatus_c;
		//	 PhyTimeDisableEventTimeout();
			 smacState = mSmacStateIdle_c;
			 MCPSDataIndication(smacProccesPacketPtr.smacRxPacketPointer);			 
		 }
	}
	 else
	 {
		 smacProccesPacketPtr.smacRxPacketPointer->rxStatus = rxSuccessStatus_c;
		// PhyTimeDisableEventTimeout();  
		 smacState = mSmacStateIdle_c;
		 MCPSDataIndication(smacProccesPacketPtr.smacRxPacketPointer);
		 
	 }
}

void    PhyPlmeCcaConfirm(bool_t channelInUse)
{
	 (void)channelInUse;
}

void    PhyPlmeEdConfirm(uint8_t energyLevel)
{
	 (void)energyLevel;
}

void    PhyPlmeRxSFDDetectIndication(void)
{

}

void    PhyPlmeRxPHRErrorIndication(void)
{
	uint8_t u8PhyRes = 0;
	 if (mSmacStateReceiving_c == smacState)
	 {
		 u8PhyRes = PhyPlmeRxRequest((phyPacket_t *)&(smacProccesPacketPtr.smacRxPacketPointer->u8DataLength), &smacLastDataRxParams, gImmediateAction_c);
		 if(!u8PhyRes)
		 {
			 if(mSmacTimeoutAsked)
			 {
				 SmacSetRxTimeout((zbClock24_t)smacTimeout);
			 }
			 smacState = mSmacStateReceiving_c;
		 
		 }
	 }
}

void    PhyPlmeRxCRCErrorIndication(void)
{
	uint8_t u8PhyRes = 0;
	 if (mSmacStateReceiving_c == smacState)
	 {
		 u8PhyRes = PhyPlmeRxRequest((phyPacket_t *)&(smacProccesPacketPtr.smacRxPacketPointer->u8DataLength), &smacLastDataRxParams, gImmediateAction_c);
		 if(!u8PhyRes)
		 {
			 if(mSmacTimeoutAsked)
			 {
				 SmacSetRxTimeout((zbClock24_t)smacTimeout);
				 
			 }
			 smacState = mSmacStateReceiving_c;
			 
		 }
	 }
}

void    PhyPlmeUnlockTx(void)
{
	 
}

void    PhyPlmeUnlockRx(void)
{
	 
}

void PhyPlmeB2BIndication(void)
{
	
}

//PHY-TIMING
void    PhyTimeTimeoutIndication(void)
{
	smacProccesPacketPtr.smacRxPacketPointer->rxStatus = rxTimeOutStatus_c;
	MCPSDataIndication(smacProccesPacketPtr.smacRxPacketPointer);
	smacState= mSmacStateIdle_c;
}
